<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\Common\Cache;

if (!defined('ABSPATH')) exit;


/**
 * Array cache driver.
 *
 * @link   www.doctrine-project.org
 * @since  2.0
 * @author Benjamin Eberlei <kontakt@beberlei.de>
 * @author Guilherme Blanco <guilhermeblanco@hotmail.com>
 * @author Jonathan Wage <jonwage@gmail.com>
 * @author Roman Borschel <roman@code-factory.org>
 * @author David Abdemoulaie <dave@hobodave.com>
 */
class ArrayCache extends \MailPoetVendor\Doctrine\Common\Cache\CacheProvider
{
    /**
     * @var array[] $data each element being a tuple of [$data, $expiration], where the expiration is int|bool
     */
    private $data = [];
    /**
     * @var int
     */
    private $hitsCount = 0;
    /**
     * @var int
     */
    private $missesCount = 0;
    /**
     * @var int
     */
    private $upTime;
    /**
     * {@inheritdoc}
     */
    public function __construct()
    {
        $this->upTime = \time();
    }
    /**
     * {@inheritdoc}
     */
    protected function doFetch($id)
    {
        if (!$this->doContains($id)) {
            $this->missesCount += 1;
            return \false;
        }
        $this->hitsCount += 1;
        return $this->data[$id][0];
    }
    /**
     * {@inheritdoc}
     */
    protected function doContains($id)
    {
        if (!isset($this->data[$id])) {
            return \false;
        }
        $expiration = $this->data[$id][1];
        if ($expiration && $expiration < \time()) {
            $this->doDelete($id);
            return \false;
        }
        return \true;
    }
    /**
     * {@inheritdoc}
     */
    protected function doSave($id, $data, $lifeTime = 0)
    {
        $this->data[$id] = [$data, $lifeTime ? \time() + $lifeTime : \false];
        return \true;
    }
    /**
     * {@inheritdoc}
     */
    protected function doDelete($id)
    {
        unset($this->data[$id]);
        return \true;
    }
    /**
     * {@inheritdoc}
     */
    protected function doFlush()
    {
        $this->data = [];
        return \true;
    }
    /**
     * {@inheritdoc}
     */
    protected function doGetStats()
    {
        return [\MailPoetVendor\Doctrine\Common\Cache\Cache::STATS_HITS => $this->hitsCount, \MailPoetVendor\Doctrine\Common\Cache\Cache::STATS_MISSES => $this->missesCount, \MailPoetVendor\Doctrine\Common\Cache\Cache::STATS_UPTIME => $this->upTime, \MailPoetVendor\Doctrine\Common\Cache\Cache::STATS_MEMORY_USAGE => null, \MailPoetVendor\Doctrine\Common\Cache\Cache::STATS_MEMORY_AVAILABLE => null];
    }
}
