<?php

namespace MailPoetVendor;

if (!defined('ABSPATH')) exit;


/*
 * This file is part of SwiftMailer.
 * (c) 2004-2009 Chris Corbyn
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
/**
 * The EventDispatcher which handles the event dispatching layer.
 *
 * @author Chris Corbyn
 */
class Swift_Events_SimpleEventDispatcher implements \MailPoetVendor\Swift_Events_EventDispatcher
{
    /** A map of event types to their associated listener types */
    private $eventMap = [];
    /** Event listeners bound to this dispatcher */
    private $listeners = [];
    /** Listeners queued to have an Event bubbled up the stack to them */
    private $bubbleQueue = [];
    /**
     * Create a new EventDispatcher.
     */
    public function __construct()
    {
        $this->eventMap = ['Swift_Events_CommandEvent' => 'MailPoetVendor\\Swift_Events_CommandListener', 'Swift_Events_ResponseEvent' => 'MailPoetVendor\\Swift_Events_ResponseListener', 'Swift_Events_SendEvent' => 'MailPoetVendor\\Swift_Events_SendListener', 'Swift_Events_TransportChangeEvent' => 'MailPoetVendor\\Swift_Events_TransportChangeListener', 'Swift_Events_TransportExceptionEvent' => 'MailPoetVendor\\Swift_Events_TransportExceptionListener'];
    }
    /**
     * Create a new SendEvent for $source and $message.
     *
     * @return Swift_Events_SendEvent
     */
    public function createSendEvent(\MailPoetVendor\Swift_Transport $source, \MailPoetVendor\Swift_Mime_SimpleMessage $message)
    {
        return new \MailPoetVendor\Swift_Events_SendEvent($source, $message);
    }
    /**
     * Create a new CommandEvent for $source and $command.
     *
     * @param string $command      That will be executed
     * @param array  $successCodes That are needed
     *
     * @return Swift_Events_CommandEvent
     */
    public function createCommandEvent(\MailPoetVendor\Swift_Transport $source, $command, $successCodes = [])
    {
        return new \MailPoetVendor\Swift_Events_CommandEvent($source, $command, $successCodes);
    }
    /**
     * Create a new ResponseEvent for $source and $response.
     *
     * @param string $response
     * @param bool   $valid    If the response is valid
     *
     * @return Swift_Events_ResponseEvent
     */
    public function createResponseEvent(\MailPoetVendor\Swift_Transport $source, $response, $valid)
    {
        return new \MailPoetVendor\Swift_Events_ResponseEvent($source, $response, $valid);
    }
    /**
     * Create a new TransportChangeEvent for $source.
     *
     * @return Swift_Events_TransportChangeEvent
     */
    public function createTransportChangeEvent(\MailPoetVendor\Swift_Transport $source)
    {
        return new \MailPoetVendor\Swift_Events_TransportChangeEvent($source);
    }
    /**
     * Create a new TransportExceptionEvent for $source.
     *
     * @return Swift_Events_TransportExceptionEvent
     */
    public function createTransportExceptionEvent(\MailPoetVendor\Swift_Transport $source, \MailPoetVendor\Swift_TransportException $ex)
    {
        return new \MailPoetVendor\Swift_Events_TransportExceptionEvent($source, $ex);
    }
    /**
     * Bind an event listener to this dispatcher.
     */
    public function bindEventListener(\MailPoetVendor\Swift_Events_EventListener $listener)
    {
        foreach ($this->listeners as $l) {
            // Already loaded
            if ($l === $listener) {
                return;
            }
        }
        $this->listeners[] = $listener;
    }
    /**
     * Dispatch the given Event to all suitable listeners.
     *
     * @param string $target method
     */
    public function dispatchEvent(\MailPoetVendor\Swift_Events_EventObject $evt, $target)
    {
        $this->prepareBubbleQueue($evt);
        $this->bubble($evt, $target);
    }
    /** Queue listeners on a stack ready for $evt to be bubbled up it */
    private function prepareBubbleQueue(\MailPoetVendor\Swift_Events_EventObject $evt)
    {
        $this->bubbleQueue = [];
        $evtClass = \get_class($evt);
        foreach ($this->listeners as $listener) {
            if (\array_key_exists($evtClass, $this->eventMap) && $listener instanceof $this->eventMap[$evtClass]) {
                $this->bubbleQueue[] = $listener;
            }
        }
    }
    /** Bubble $evt up the stack calling $target() on each listener */
    private function bubble(\MailPoetVendor\Swift_Events_EventObject $evt, $target)
    {
        if (!$evt->bubbleCancelled() && ($listener = \array_shift($this->bubbleQueue))) {
            $listener->{$target}($evt);
            $this->bubble($evt, $target);
        }
    }
}
